/**
  ******************************************************************************
  * @file    cp_payload.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Implements the comm protocol common code for both client and server ends
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "project.h"

#include "cp.h"

// INPUT SIDE

#define EXTRACTLONG(target, ptr) target =  (((ptr[0]) << 24) + ((ptr[1]) << 16) + ((ptr[2]) << 8) + (ptr[3]) ); ptr += sizeof (unsigned long);

#define STORELONG(ptr,longval) {ptr[0] = (longval >> 24); ptr[1] = (longval >> 16); ptr[2] = (longval >> 8); ptr[3] = (unsigned char)longval; ptr += sizeof(unsigned long); };

/** @brief Checks the syntactic correctness of a packet according to the protocol specification layer 5 (TLV structure).
 *
 *  @param p_Buf Packet to analyze
 *  @param p_BufLen Length of chunk
 *  @param p_DynReceiver (output parameter) receives a chained list of dynamic tag structures
 *  @param p_TagBitmask receives bitwise information which of the key tags are present (see enumerator TAG_BITMASK_XXX)
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * called in the context of a receiver task.
 */

CP_STATUSCODE cp_SyntaxCheck(unsigned char *p_Buf,unsigned long p_BufLen,CP_DYNTAGSTRUCT **p_DynReceiver,unsigned short *p_TagBitmask)
{
    unsigned char *a_Running,*a_End;
    CP_STATUSCODE a_Result = CP_STATUSCODE_CORRUPT;
    *p_DynReceiver = 0; // default
    *p_TagBitmask = 0;
    // 1. Sanity check to prevent the future traversal routine from crashing w/ too short packets
    if (p_BufLen > sizeof(unsigned long)*2)
    {
        a_Running = p_Buf;
        a_End = &p_Buf[p_BufLen];
        while (a_Running < a_End)
        {
            CP_DYNTAGSTRUCT *a_NextDynTagPtr=0;
            unsigned long a_NextTag,a_NextLen;
            EXTRACTLONG(a_NextTag,a_Running);
            switch (a_NextTag)
            {
                case TAG_TELEGID: SET_BIT_IN_BITMASK(*p_TagBitmask,TAG_BITMASK_ID_PRESENT);  break; 
                case TAG_RESPONSE: SET_BIT_IN_BITMASK(*p_TagBitmask,TAG_BITMASK_RESPONSE_PRESENT); break;
                case TAG_AUTHREQUEST: SET_BIT_IN_BITMASK(*p_TagBitmask,TAG_BITMASK_AUTH_PRESENT); break;
            };       
            EXTRACTLONG(a_NextLen,a_Running);
            if (a_NextLen <= (sizeof(unsigned long)*2)) 
            {
                cp_DeleteTagChain(p_DynReceiver);            
                return CP_STATUSCODE_CORRUPT; // sanity, otherwise an illformed packet may loop infinitely. TODO: rule out impossibly big lengths
            }
            else
                a_NextLen -= (sizeof(unsigned long)*2);
            if (a_NextTag == TAG_TELEGID)
                a_Result = CP_STATUSCODE_SUCCESS;
            // record current tag to pass back
            if (cp_DynTagFromTLV(a_NextTag,a_NextLen,a_Running,&a_NextDynTagPtr) !=  CP_STATUSCODE_SUCCESS)
            {
                cp_DeleteTagChain(p_DynReceiver);
                return CP_STATUSCODE_CORRUPT;
            }
            cp_InsertDynTagAtEnd(p_DynReceiver,a_NextDynTagPtr);                        
            // skip to next TLV in teleg
            a_Running += a_NextLen; 
        }
        if (a_Running > a_End) // overrun!
        {
            cp_DeleteTagChain(p_DynReceiver);        
            return CP_STATUSCODE_CORRUPT;
        }
    }
    return a_Result;    
}

/** @brief Determines whether a communication has passed the authentication phase.
 *
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE CP_PeerIsAuthenticated(PCP_PROCESSOR p_Processor)
{
    unsigned long a_AuthStatus = p_Processor->m_AuthStatus;
    if (a_AuthStatus & CP_AUTHSTATUS_FLAG_PEERMUSTAUTHENTICATE)
    {
        if (!(a_AuthStatus & CP_AUTHSTATUS_FLAG_AUTHRECEIVED))
            return CP_STATUSCODE_AUTHFAIL;
    }
    return CP_STATUSCODE_SUCCESS;
}

/** @brief Determines if a receives packet is an "authentication" packet
 *
 *  @param p_DynTagStruct chained list of dynamic tag structures
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * called in the context of a receiver task.
 */

void cp_TestForAuthPacket(CP_DYNTAGSTRUCT *p_DynTagStruct,PCP_PROCESSOR p_Processor)
{
    while (p_DynTagStruct)
    {
        if (p_DynTagStruct->m_Tag == TAG_AUTHREQUEST) 
        {
            if (cp_CB_AuthenticateClient(p_DynTagStruct) == CP_STATUSCODE_SUCCESS)
                p_Processor->m_AuthStatus |= CP_AUTHSTATUS_FLAG_AUTHRECEIVED;
            break;
        }
        p_DynTagStruct = p_DynTagStruct->m_Next;
    }
}

/** @brief Preprocesses a packet from the input stream. Syntactic check is done. Passes the packet on to the application  routine.
 *
 *  @param p_DynTagStruct chained list of dynamic tag structures
 *  @param p_Processor communication processor
 *  @param p_OrgSeq (output parameter) receives the sequence number from the original packet
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * called in the context of a receiver task.
 */

CP_STATUSCODE cp_DissectIncomingPacket(CP_DYNTAGSTRUCT *p_DynTagStruct,PCP_PROCESSOR p_Processor,unsigned long *p_OrgSeq)
{
    CP_STATUSCODE a_Return = CP_STATUSCODE_CORRUPT;  // default in case we have a completly defunkt packet w/ ID only but no other tag...
    while (p_DynTagStruct)
    {
        if (p_DynTagStruct->m_Tag == TAG_TELEGID)
            *p_OrgSeq = htonl(*((unsigned long *)p_DynTagStruct->m_MarshalledVal));
        else // again, this assumes packets w/ exactly two TLVs - id and payload - included (response tags have been filtered out before).
            a_Return = cp_EvaluateIncomingPacket(p_Processor,p_DynTagStruct);
        p_DynTagStruct = p_DynTagStruct->m_Next;
    }
    return a_Return;
}


// OUTPUT SIDE

/** @brief Layer 4 output routine of the protocol. Tucks the sequence number up front the packet.
 *
 * 
 *  @param p_Tag Tag for packet
 *  @param p_Len Len for packet
 *  @param p_Val Value for packet
 *  @param p_SequenceNo Sequence No. to include in packet
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * to keep things simple, we only allow one tag after ID.
 */

static CP_STATUSCODE cp_ConvertOutPacketAndEmit(unsigned long p_Tag,unsigned long p_Len,unsigned char *p_Val, unsigned long p_SequenceNo, PCP_PROCESSOR p_Processor)
{
    CP_STATUSCODE a_Result = CP_STATUSCODE_NOMEM;
    unsigned long a_PacketLen = p_Len + 5* sizeof(unsigned long);
    unsigned char *a_Packet = (unsigned char *)pvPortMalloc(a_PacketLen);
    if (a_Packet)
    {
        unsigned char *a_Running = a_Packet; 
        // tuck the ID tag at the beginning.
        STORELONG(a_Running,TAG_TELEGID);
        STORELONG(a_Running,3*sizeof(unsigned long));
        STORELONG(a_Running,p_SequenceNo); 
        STORELONG(a_Running,p_Tag);
        STORELONG(a_Running,(p_Len+2*sizeof(unsigned long)));
        memcpy(a_Running,p_Val,p_Len);
        a_Result = CP_FrameAndEmitPacket(a_Packet,a_PacketLen,p_Processor); 
        vPortFree(a_Packet);
    }
    return a_Result;
}

/** @brief Wrapper for cp_ConvertOutPacketAndEmit with a reduced parameter set.
 *
 * 
 *  @param p_SequenceNo Sequence No. to include in packet
 *  @param p_Tag Tag for packet
 *  @param p_TagValue Single integer value for packet
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 */

CP_STATUSCODE cp_SendSingleIntegerTagPacket(unsigned long p_SequenceNo,unsigned long p_Tag,unsigned long p_TagValue, PCP_PROCESSOR p_Processor)
{
    unsigned long aModified = htonl(p_TagValue);
    return cp_ConvertOutPacketAndEmit(p_Tag,sizeof(unsigned long),(unsigned char *)&aModified,p_SequenceNo,p_Processor);
}

/** @brief Sends a packet response.
 *
 * 
 *  @param p_SequenceNo Sequence No. to include in packet
 *  @param p_ResponseCode Response to send to the peer
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Called in the context of the receiver task (responses are always inlined)
 */

CP_STATUSCODE cp_RespondPacket(unsigned long p_SequenceNo,unsigned long p_ResponseCode, PCP_PROCESSOR p_Processor)
{
    unsigned long aModified = htonl(p_ResponseCode);
    return cp_ConvertOutPacketAndEmit(TAG_RESPONSE,sizeof(unsigned long),(unsigned char *)&aModified,p_SequenceNo,p_Processor);
}

// this is a little cumbersome; we need to reparse the dynamic structure here to find the data necessary to evaluate the response...
// called in the context of the receiver task.

#define BITMASK_RESPONSECODEGOOD (1 << 0)
#define BITMASK_SEQDECODED       (1 << 1)
#define BITMASK_SEQMATCHGOOD     (1 << 2)

/** @brief Processes a response packet from the peer.
 *
 * 
 *  @param p_DynTagStruct chained list of dynamic tag structures
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Called in the context of the receiver task
 * This matches the sequence number included in a response against the sequence number of the last packet emitted and removes the packet from the
 * out queue if the response code is ok and the sequence numbers match.
 */

CP_STATUSCODE CP_ProcessResponsePacket(CP_DYNTAGSTRUCT *p_DynTagStruct,PCP_PROCESSOR p_Processor)
{
    CP_MSGPUMPINFO a_CurrentPacketAwaitingACK;
    unsigned long a_ResponseOK = 0;
    unsigned long a_SeqOfIncomingResponse;
    CP_STATUSCODE a_Return = CP_STATUSCODE_NODATA;
    CP_PRILEVELS a_Pri;
    // first check response code, only proceed is response ok
    while (p_DynTagStruct)
    {
        if ((p_DynTagStruct->m_Tag == TAG_RESPONSE) && (*((unsigned long *)p_DynTagStruct->m_MarshalledVal) == htonl(CP_RESPONSE_ACK)))
        {
            a_ResponseOK |= BITMASK_RESPONSECODEGOOD;
        }
        if (p_DynTagStruct->m_Tag == TAG_TELEGID)
        {
            a_SeqOfIncomingResponse = htonl(*((unsigned long *)p_DynTagStruct->m_MarshalledVal));
            a_ResponseOK |= BITMASK_SEQDECODED;  // sanity. Is always true becuase the packet has been preprocessed here.
        }
        p_DynTagStruct = p_DynTagStruct->m_Next;
    }
    if (!(a_ResponseOK | BITMASK_RESPONSECODEGOOD))
        return CP_STATUSCODE_PACKET_NOT_ACKED;
    if (!(a_ResponseOK | BITMASK_SEQDECODED))
        return CP_STATUSCODE_CORRUPT;
    // determine outbound packet to match response against. We need mutual exclusion here so that concurrent Timeouts don't
    // cause a mismatch.
    CP_BeginAtomic(p_Processor->m_CommQueue);
    a_Pri = p_Processor->m_CommQueue->m_QueueCurrentlySending;
    if (a_Pri != CP_NOQUEUE)
    {
        do  // attempt to find the packet on the queues of all priority levels. This is to address the race condition where packets of different
            // priority levels are concurrently waiting for acknowledgement.
        {
            a_Return = CP_PeekPriQueue(p_Processor->m_CommQueue,&a_CurrentPacketAwaitingACK,&a_Pri);    
            if (a_Return == CP_STATUSCODE_SUCCESS)
            {
                a_Return = CP_STATUSCODE_CORRUPT; // default now. Overridden if everything is good.
                // a few sanity checks.
                if ((a_CurrentPacketAwaitingACK.m_Cmd == CP_CMD_SEND_SINGLE_PAYLOAD_TLV) && 
                    (a_CurrentPacketAwaitingACK.m_u.singleTLVpayloaddata.m_Status == TELEG_SENT))
                {
                    a_Return = CP_STATUSCODE_PACKET_NOT_ACKED; // default now. Overridden if everything is good.
                    if (a_CurrentPacketAwaitingACK.m_SeqNo == a_SeqOfIncomingResponse)
                    {
                        CP_RemovedPeekedPriQueue(p_Processor->m_CommQueue,p_Processor->m_CommQueue->m_QueueCurrentlySending);
                        p_Processor->m_CommQueue->m_QueueCurrentlySending = CP_NOQUEUE;
                        a_Return = CP_STATUSCODE_SUCCESS;
                    } // sequence numbers matched!
                } // sanity checks on the current queue item
            } // could retrieve the queue item
            a_Pri++;  // try next priority level
        } while ((a_Return = CP_STATUSCODE_SUCCESS) && (a_Pri < CP_PRIQUEUE_MAXPRIORITIES)); // iteration over the queues 
    } // There was a pending packet (priority in the comm queue was not CP_NOQUEUE). Race conditions can time out a packet right before the ACK is received,
      // so this is a valid scenario.
    CP_EndAtomic(p_Processor->m_CommQueue);
    return a_Return;
}

/** @brief marks a packet as emitted and waiting for an ACK.
 *
 * 
 *  @param p_CurrentComm emitted packet
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Called in the context of a server or client task
 */

CP_STATUSCODE CP_MarkPacketAsSentAndAwaitingACK(PCP_MSGPUMPINFO p_CurrentComm,PCP_PROCESSOR p_Processor)
{
    // change status and put back in queue. Watch out for possible race conditions if the response comes in really fast!
    CP_BeginAtomic(p_Processor->m_CommQueue);
    CP_RemovedPeekedPriQueue(p_Processor->m_CommQueue,p_CurrentComm->m_Pri);
    p_CurrentComm->m_u.singleTLVpayloaddata.m_Status = TELEG_SENT;
    if (xQueueSendToFront(p_Processor->m_CommQueue->m_Queues[p_CurrentComm->m_Pri],p_CurrentComm,0) != pdTRUE) STATIC_SANITY_CHECK;
    CP_EndAtomic(p_Processor->m_CommQueue);
    return CP_STATUSCODE_SUCCESS;
}

#define TELEGTIMEOUTTICKCOUNT 5000 

/** @brief role- independent execution loop of the communication.
 *
 * 
 *  @param p_CurrentComm emitted packet
 *  @param p_Processor communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Terminates if and only if the communication channel is broken
 * Implemented as a message pump.
 *
 */

CP_STATUSCODE cp_HandleEstablishedComm(PCP_PROCESSOR p_Processor)
{
    CP_MSGPUMPINFO a_CurrentComm;
    CP_PRILEVELS a_Pri;
    unsigned long a_StillInProgress = 1;
    while (a_StillInProgress)
    {
        CP_BeginAtomic(p_Processor->m_CommQueue); // we compete against the receiver task...
        a_Pri = CP_NOQUEUE; // means: select highest priority element.
        switch (CP_PeekPriQueue(p_Processor->m_CommQueue,&a_CurrentComm,&a_Pri))
        {
            case CP_STATUSCODE_SUCCESS:
            {
                a_CurrentComm.m_Pri = a_Pri;  // save the returned pri for later processing. Important: This will modify only this copy, not the MSGPUMP structure in the queue! 
                switch (a_CurrentComm.m_Cmd)
                {
                    // it CAN happen that this telegram overlaps with an already scheduled telegram, namely, when an application has scheduled a higher priority
                    // Telegram while a lower pri Telegram is still waiting for an acknowledgment. This is perfectly ok; the acknowledgment mechanism will attempt to find
                    // a telegram awaiting ACK on any queue and still match it.
                    case CP_CMD_SEND_SINGLE_PAYLOAD_TLV: // command to be removed by ACK reception
                    {
                        switch (a_CurrentComm.m_u.singleTLVpayloaddata.m_Status)
                        {
                            case TELEG_SCHEDULED:
                            // this predicate to prevent a server from emitting pending packets until the client has authenticated! For the client this is always true so
                            // the client will send a waiting auth packet. 
                            if (CP_PeerIsAuthenticated(p_Processor) == CP_STATUSCODE_SUCCESS)
                            {
                                p_Processor->m_CommQueue->m_QueueCurrentlySending = a_CurrentComm.m_Pri;
                                a_CurrentComm.m_u.singleTLVpayloaddata.m_ScheduleTime = xTaskGetTickCount(); // to allow for timeouts
                                if (cp_SendSingleIntegerTagPacket(a_CurrentComm.m_SeqNo,a_CurrentComm.m_u.singleTLVpayloaddata.m_Tag,a_CurrentComm.m_u.singleTLVpayloaddata.m_Value,p_Processor) == CP_STATUSCODE_SUCCESS)
                                {
                                    CP_MarkPacketAsSentAndAwaitingACK(&a_CurrentComm,p_Processor);
                                }  // send successful
                                else  // TESTTESTTEST
                                {
                                    p_Processor->m_CommQueue->m_QueueCurrentlySending = CP_NOQUEUE;
                            
                                // communication error. Do not terminate the connection here; the reader half of the communicator will do that (it does a periodic read attempt)
                                } // send not successful
                            } // the connection state is past authentication.
                            break;
                            case TELEG_SENT:
                                if ((xTaskGetTickCount() - a_CurrentComm.m_u.singleTLVpayloaddata.m_ScheduleTime) >= TELEGTIMEOUTTICKCOUNT)
                                {
                                    // simply marking it as scheduled will start the thing again next turn!
                                    CP_RemovedPeekedPriQueue(p_Processor->m_CommQueue,a_CurrentComm.m_Pri);
                                    a_CurrentComm.m_u.singleTLVpayloaddata.m_Status = TELEG_SCHEDULED;
                                    if (xQueueSendToFront(p_Processor->m_CommQueue->m_Queues[a_CurrentComm.m_Pri],&a_CurrentComm,0) != pdTRUE) STATIC_SANITY_CHECK;
                                    p_Processor->m_CommQueue->m_QueueCurrentlySending = CP_NOQUEUE;
                                }
                            break;
                            case TELEG_ACKED: // so far won't happen
                            break;
                        }; // switch on teleg state
                    };
                    break; // case CP_CMD_SEND_SINGLE_PAYLOAD_TLV
                    case CP_CMD_TERMINATE_CONNECTION:
                        a_StillInProgress = 0; 
                        CP_RemovedPeekedPriQueue(p_Processor->m_CommQueue,a_Pri); // this removes the TERMINATE_CONNECTION command, not pending telegrams!
                    break;
                }; // switch on command
            }; // case CP_STATUSCODE_SUCCESS
            break;
            case CP_STATUSCODE_NODATA:
            break;
            default: 
                break;
        }; // switch on peek on queue
        CP_EndAtomic(p_Processor->m_CommQueue);        
        vTaskDelay(2); // this prevents the task from getting stuck in a CPU bound loop.
    }; // while (aStillInProgress)
    return CP_STATUSCODE_CONNLOSS;
}
